
//	=================================================================
//	ROUTINES.C					26-09-2001 Arttu Soininen
//
//	Example code fragments for:
//	 - reading in TerraScan binary files
//	=================================================================

typedef unsigned char BYTE ;
typedef unsigned short USHORT ;
typedef unsigned int UINT ;

//	TerraScan binary file point record (ver 20010712, 20010129, 970404)

typedef struct {
	BYTE	Code ;		//	Classification code 0-255
	BYTE	Line ;		//	Flightline number 0-255
	USHORT	EchoInt ;	//	Intensity bits 0-13, echo bits 14-15
	long	X ;			//	Coordinates
	long	Y ;
	long	Z ;
} ScanRow;

//	TerraScan binary file point record ver 20020715

typedef struct {
	Point3d	Pnt ;		//	Coordinates
	BYTE	Code ;		//	Classification code
	BYTE	Echo ;		//	Echo information
	BYTE	Flag ;		//	Runtime flag (view visibility)
	BYTE	Mark ;		//	Runtime flag
	USHORT	Line ;		//	Flightline number
	USHORT	Intensity ;	//	Intensity value
} ScanPnt ;

//	TerraScan binary file header

typedef struct {
	int		HdrSize ;	//	sizeof(ScanHdr)
	int		HdrVersion;	//	Version number 20020715, 20010712, 20010129 or 970404
	int		Tunniste ;	//	Always 970401	
	char	Magic[4];	//	CXYZ
	int		PntCnt ;	//	Number of points stored		
	int		Units ;		//	Units per meter = subpermast * uorpersub
	double	OrgX ;		//	Global origin
	double	OrgY ;
	double	OrgZ ;
	int		Time ;		//	32 bit integer time stamps appended to points
	int		Color ;		//	Color values appended to points
} ScanHdr ;

//	Undefined time stamp

#define SEC_UNDEFINED	0xFFFFFFFF

/*-------------------------------------------------------------------
	Attemp reading TerraScan binary file header information
	from file Name.
	
	Return  1 on success.
	Return  0 if could not open file.
	Return -1 if file is of wrong format.
*/

int		ScanGetHeader( ScanHdr *Hdr, char *Name)
{
	FILE	*File ;
	int		Ok ;

	File = fopen( Name, "rb") ;
	if (!File)									return (0) ;

	Ok = ScanReadHdr( Hdr, File) ;
	fclose( File) ;
	if (!Ok)									return (-1) ;

	return (1) ;
}

/*-------------------------------------------------------------------
	Read binary scan points from file Path into table Tbl.
	Table has been allocated for Cnt points.

	Return number of points read.
	Return  0 if not a valid binary file.
	Return -1 if could not open file.
*/

int     ScanReadBinary( ScanPnt *Tbl, UINT *Sec, RgbClr *Clr, int Cnt, char *Path)
{
	FILE	*File ;
	ScanHdr	Hdr ;
	ScanRow	Mit ;
	ScanPnt	*P ;
	BYTE	Rgb[4] ;
	UINT	Time = SEC_UNDEFINED ;
	int		Ret  = 0 ;
	int		Ok ;

	//	Open file for reading

	File = fopen( Path, "rb") ;
	if (!File)									return (-1) ;

	if (!ScanReadHdr( &Hdr, File))
		Hdr.PntCnt = 0 ;

	//	Use white if no color in file

	Rgb[0] = 255 ;
	Rgb[1] = 255 ;
	Rgb[2] = 255 ;

	while ((Ret < Hdr.PntCnt) && (Ret < Cnt)) {
		P = Tbl + Ret ;
		if (Hdr.HdrVersion == 20020715) {
			Ok = fread( P, sizeof(*P), 1, File) ;
			if (Ok != 1)							break ;
		}
		else {
			Ok = fread( &Mit, sizeof(Mit), 1, File) ;
			if (Ok != 1)							break ;

			//	Convert point information

			P->Pnt.x     = Mit.X ;
			P->Pnt.y     = Mit.Y ;
			P->Pnt.z     = Mit.Z ;
			P->Code      = Mit.Code ;
			P->Line      = Mit.Line ;
			P->Intensity = Mit.EchoInt & 0x3FFF ;
			P->Echo      = (Mit.EchoInt >> 14) ;
		}

		//	Read time stamp if given

		if (Hdr.Time)
			fread( &Time, sizeof(UINT), 1, File) ;
		if (Sec)
			Sec[Ret] = Time ;

		//	Read color value if given

		if (Hdr.Color)
			fread( Rgb, 4, 1, File) ;
		if (Clr) {
			Clr[Ret].red   = Rgb[0] ;
			Clr[Ret].blue  = Rgb[1] ;
			Clr[Ret].green = Rgb[2] ;
		}
		Ret++ ;
	}
	fclose( File) ;

	return (Ret) ;
}

/*-------------------------------------------------------------------
	Read TerraScan binary file header and check validity.

	Return 1 if valid header was found.
	Return 0 if not TerraScan binary file.
*/

int		ScanReadHdr( ScanHdr *Hp, FILE *File)
{
	ScanHdr	Hdr ;
	int		CopySize ;
	int		Val ;
	int		Ok ;

	Ok  = fread( &Hdr, sizeof(ScanHdr), 1, File) ;
	if (Ok != 1)								return (0) ;

	Val = ScanHeaderValid( &Hdr) ;
	if (!Val)									return (0) ;

	if (Hdr.HdrSize != sizeof(ScanHdr)) {
		CopySize = sizeof(ScanHdr) ;
		if (Hdr.HdrSize < CopySize)
			CopySize = Hdr.HdrSize ;
		memset( Hp, 0, sizeof(ScanHdr)) ;
		memcpy( Hp, &Hdr, CopySize) ;
		fseek( File, Hdr.HdrSize, SEEK_SET) ;
	}
	else {
		*Hp = Hdr ;
	}
	return (1) ;
}

/*-------------------------------------------------------------------
	Check if H is a valid TerraScan binary file header.

	Return 5 if valid header newer than 15.07.2002.
	Return 4 if valid header version 15.07.2002.
	Return 3 if valid header version 12.07.2001.
	Return 2 if valid header version 29.01.2001.
	Return 1 if valid header version 04.04.1997.
	Return 0 if not valid.
*/

int		SrvScanHeaderValid( ScanHdr *H)
{
	int		V ;

	if (H->Tunniste != 970401)					return (0) ;
	if (memcmp(H->Magic,"CXYZ",4))				return (0) ;

	V = H->HdrVersion ;
	if (V == 970404)							return (1) ;
	if (V == 20010129)							return (2) ;
	if (V == 20010712)							return (3) ;
	if (V == 20020715)							return (4) ;
	if ((V > 20020715) && (V < 20051231))		return (5) ;

	return (0) ;
}
